<?php
/**
* 2007-2020 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    PrestaShop SA <contact@prestashop.com>
*  @copyright 2007-2020 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

if (!defined('_PS_VERSION_')) {
    exit;
}

class Wiserreview extends Module
{
    const _MPrefix = 'WR_';

    const JS_VAR = 'wrConfigVar';

    public $script = '';

    public $notification = [];

    public function __construct()
    {
        $this->name = 'wiserreview';
        $this->tab = 'administration';
        $this->version = '1.0.0';
        $this->author = 'By WiserNotify';
        $this->bootstrap = true;
        parent::__construct();
        $this->displayName = $this->l('WiserReview by WiserNotify');
        $this->description = $this->l('WiserReview: Automate, Manage, and Display Your Customer Reviews with Ease!');
        $this->confirmUninstall = $this->l('Are you sure to uninstall ?');
        $this->ps_versions_compliancy = array('min' => '1.7', 'max' => _PS_VERSION_);
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        // Register hooks
        $this->registerHook('displayWiserReviewNudge');
        $this->registerHook('displayWiserReviewProductListReviews');
        $this->registerHook('displayWiserReviewFooterProduct');
        $this->registerHook('displayWiserReviewProductPriceBlock');
        $this->registerHook('displayHeader');
        $this->registerHook('displaybackOfficeHeader');
        $this->registerHook('displayProductListReviews');
        $this->registerHook('displayFooterProduct');
        $this->registerHook('displayProductPriceBlock');
        $this->registerHook('displayProductAdditionalInfo');
        $this->registerHook('actionValidateOrder');

        $shops = Shop::getShops();
        foreach ($shops as $shop) {
            $id_shop = (int)$shop['id_shop'];

            Configuration::updateValue(self::_MPrefix.'LIVE_MODE_'.$id_shop, false);
            Configuration::updateValue(self::_MPrefix.'API_KEY_'.$id_shop, '');
            // 
            Configuration::updateValue(self::_MPrefix.'PRODUCT_SECTION_'.$id_shop, false);
            Configuration::updateValue(self::_MPrefix.'COLLECTION_SECTION_'.$id_shop, false);
            Configuration::updateValue(self::_MPrefix.'PRODUCT_REVIEW_SECTION_'.$id_shop, false);
            Configuration::updateValue(self::_MPrefix.'REVIEW_NUDGES_'.$id_shop, false);
            Configuration::updateValue(self::_MPrefix.'WISER_REVIEW_TI_'.$id_shop, '');
            Configuration::updateValue(self::_MPrefix.'WISER_AUTOMATION_ID_'.$id_shop, '');
            // 
            Configuration::updateValue(self::_MPrefix.'WISER_HEADER_PXL_'.$id_shop, '');
            Configuration::updateValue(self::_MPrefix.'WISER_CONTENT_NUDGE_'.$id_shop, '');
            Configuration::updateValue(self::_MPrefix.'WISER_CONTENT_RATING_'.$id_shop, '');
            Configuration::updateValue(self::_MPrefix.'WISER_CONTENT_REVIEW_'.$id_shop, '');
        }

        return true;
    }

    public function uninstall()
    {
        $shops = Shop::getShops();
        foreach ($shops as $shop) {
            $id_shop = (int)$shop['id_shop'];

            Configuration::deleteByName(self::_MPrefix.'LIVE_MODE_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'API_KEY_'.$id_shop);
            // 
            Configuration::deleteByName(self::_MPrefix.'PRODUCT_SECTION_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'COLLECTION_SECTION_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'PRODUCT_REVIEW_SECTION_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'REVIEW_NUDGES_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'WISER_REVIEW_TI_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'WISER_AUTOMATION_ID_'.$id_shop);
            // 
            Configuration::deleteByName(self::_MPrefix.'WISER_HEADER_PXL_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'WISER_CONTENT_NUDGE_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'WISER_CONTENT_RATING_'.$id_shop);
            Configuration::deleteByName(self::_MPrefix.'WISER_CONTENT_REVIEW_'.$id_shop);
        }
        
        // WiserReview Hooks
        $this->unregisterHook('displayWiserReviewNudge');
        $this->unregisterHook('displayWiserReviewProductListReviews');
        $this->unregisterHook('displayWiserReviewFooterProduct');
        $this->unregisterHook('displayWiserReviewProductPriceBlock');

        // PS Hooks
        $this->unregisterHook('displayHeader');
        $this->unregisterHook('displaybackOfficeHeader');
        $this->unregisterHook('displayProductListReviews');
        $this->unregisterHook('displayFooterProduct');
        $this->unregisterHook('displayProductPriceBlock');
        $this->unregisterHook('displayProductAdditionalInfo');
        $this->unregisterHook('actionValidateOrder');

        return parent::uninstall(); 
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        $shops = Shop::getShops();
        
        $adminLink = $this->context->link->getModuleLink(
            $this->name,
            'ajax',
            []
        );

        $html = '<div class="alert alert-info">' . $this->l('WiserReview module helps you collect and display product reviews, star ratings, and nudges. It also automates review requests via email to boost customer engagement and conversions.') . '
            </div>';

        // Clear previous messages to ensure only current ones are displayed
        $this->context->controller->errors = [];
        $this->context->controller->confirmations = [];
        
        if (((bool)Tools::isSubmit('submit'.$this->name)) == true) {
            $this->notification = [];

            $this->postProcess();

            // Notification Message
            foreach ($shops as $shop) {
                $id_shop = $shop['id_shop'];
                $shop_name = $shop['name'];
                if($this->notification[$id_shop]['status'] == 'success') {
                    $this->context->controller->confirmations[] = $shop_name . ' - ' . $this->notification[$id_shop]['msg'];
                } else if($this->notification[$id_shop]['status'] == 'error') {
                    $this->context->controller->errors[] = $shop_name . ' - ' . $this->notification[$id_shop]['msg'];
                }
            }
        }

        $values = $this->getConfigFormValues();
        
        $current_shop_id = $this->context->shop->id;
        $livemode = (boolean) $values[self::_MPrefix.'LIVE_MODE_'.$current_shop_id];
        $apikey = $values[self::_MPrefix.'API_KEY_'.$current_shop_id];
        $wsid = $values[self::_MPrefix.'WISER_REVIEW_TI_'.$current_shop_id];

        $ex_status = ($livemode && !empty($apikey) && !empty($wsid)) ? 'enable' : 'disable';
        
        $this->context->smarty->assign([
            'ajax_admin_link' => $adminLink,
            'ex_status' => $ex_status,
            'prefix' => self::_MPrefix ,
            'shops' => $shops,
            'fields_value' => $this->getConfigFormValues(),
        ]);

        $tplContent = '';
        $tplContent .= $this->context->smarty->fetch(_PS_MODULE_DIR_ . $this->name . '/views/templates/admin/configure.tpl');
        $tplContent .= $this->context->smarty->fetch(_PS_MODULE_DIR_ . $this->name . '/views/templates/hook/info.tpl');
        $tplContent .= $this->context->smarty->fetch(_PS_MODULE_DIR_ . $this->name . '/views/templates/hook/export-order.tpl');
        $tplContent .= $this->context->smarty->fetch(_PS_MODULE_DIR_ . $this->name . '/views/templates/hook/guide.tpl');

        return $html . $tplContent;
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        $config = [];
        foreach (Shop::getShops() as $shop) {
            $id_shop = (int)$shop['id_shop'];
            $config[self::_MPrefix.'LIVE_MODE_'.$id_shop] = (bool) Configuration::get(self::_MPrefix.'LIVE_MODE_'.$id_shop, true);
            $config[self::_MPrefix.'API_KEY_'.$id_shop] = Configuration::get(self::_MPrefix.'API_KEY_'.$id_shop);
            // 
            $config[self::_MPrefix.'PRODUCT_SECTION_'.$id_shop] = (bool) Configuration::get(self::_MPrefix.'PRODUCT_SECTION_'.$id_shop, true);
            $config[self::_MPrefix.'COLLECTION_SECTION_'.$id_shop] = (bool) Configuration::get(self::_MPrefix.'COLLECTION_SECTION_'.$id_shop, true);
            $config[self::_MPrefix.'PRODUCT_REVIEW_SECTION_'.$id_shop] = (bool) Configuration::get(self::_MPrefix.'PRODUCT_REVIEW_SECTION_'.$id_shop, true);
            $config[self::_MPrefix.'REVIEW_NUDGES_'.$id_shop] = (bool) Configuration::get(self::_MPrefix.'REVIEW_NUDGES_'.$id_shop, true);
            $config[self::_MPrefix.'WISER_REVIEW_TI_'.$id_shop] = Configuration::get(self::_MPrefix.'WISER_REVIEW_TI_'.$id_shop);
            $config[self::_MPrefix.'WISER_AUTOMATION_ID_'.$id_shop] = Configuration::get(self::_MPrefix.'WISER_AUTOMATION_ID_'.$id_shop);
            //
            $config[self::_MPrefix.'WISER_HEADER_PXL_'.$id_shop] = Configuration::get(self::_MPrefix.'WISER_HEADER_PXL_'.$id_shop);
            $config[self::_MPrefix.'WISER_CONTENT_NUDGE_'.$id_shop] = Configuration::get(self::_MPrefix.'WISER_CONTENT_NUDGE_'.$id_shop);
            $config[self::_MPrefix.'WISER_CONTENT_RATING_'.$id_shop] = Configuration::get(self::_MPrefix.'WISER_CONTENT_RATING_'.$id_shop);
            $config[self::_MPrefix.'WISER_CONTENT_REVIEW_'.$id_shop] = Configuration::get(self::_MPrefix.'WISER_CONTENT_REVIEW_'.$id_shop);
        }

        return $config;
    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $this->context->controller->errors = [];
        $this->context->controller->confirmations = [];

        $shops = Shop::getShops();
        $this->notification = [];
        
        foreach ($shops as $shop) {
            $id_shop = (int) $shop['id_shop'];

            $activeShop = (int) Tools::getValue('WISER_ACTIVE_SHOP');
            if($activeShop != $id_shop)
                continue;

            $liveMode = (bool) Tools::getValue(self::_MPrefix . 'LIVE_MODE_' . $id_shop);
            $apiKey = Tools::getValue(self::_MPrefix . 'API_KEY_' . $id_shop);
            $p_section = (bool) Tools::getValue(self::_MPrefix . 'PRODUCT_SECTION_' . $id_shop);
            $c_section = (bool) Tools::getValue(self::_MPrefix . 'COLLECTION_SECTION_' . $id_shop);
            $pr_section = (bool) Tools::getValue(self::_MPrefix . 'PRODUCT_REVIEW_SECTION_' . $id_shop);
            $r_nudges = (bool) Tools::getValue(self::_MPrefix . 'REVIEW_NUDGES_' . $id_shop);
            
            // Save API_KEY and LIVE_MODE values per shop
            Configuration::updateValue(self::_MPrefix.'LIVE_MODE_'.$id_shop, $liveMode);
            Configuration::updateValue(self::_MPrefix.'API_KEY_'.$id_shop, $apiKey);
            // 
            Configuration::updateValue(self::_MPrefix.'PRODUCT_SECTION_'.$id_shop, $p_section);
            Configuration::updateValue(self::_MPrefix.'COLLECTION_SECTION_'.$id_shop, $c_section);
            Configuration::updateValue(self::_MPrefix.'PRODUCT_REVIEW_SECTION_'.$id_shop, $pr_section);
            Configuration::updateValue(self::_MPrefix.'REVIEW_NUDGES_'.$id_shop, $r_nudges);
            
            if (!$liveMode) {
                $this->notification[$id_shop] = [
                    'status' => 'error',
                    'msg' => $this->trans('WiserReview module has been disabled. All associated features are now inactive.', [], 'Modules.WiserReview.Admin.'),
                ];
                continue;
            }

            if (empty($apiKey) && $liveMode) {
                $this->notification[$id_shop] = [
                    'status' => 'error',
                    'msg' => $this->trans('API key is required when Live Mode is enabled.', [], 'Modules.WiserReview.Admin'),
                ];
                continue;
            }

            // If API key is provided, verify it
            $tokens = $this->verifyApiTokey($apiKey);

            if ($tokens->status === 'success' && isset($tokens->data)) {
                $data = $tokens->data;

                Configuration::updateValue(self::_MPrefix . 'WISER_REVIEW_TI_' . $id_shop, $data->wsid);
                Configuration::updateValue(self::_MPrefix . 'WISER_AUTOMATION_ID_' . $id_shop, $data->automation_id);
                Configuration::updateValue(self::_MPrefix . 'WISER_HEADER_PXL_' . $id_shop, $data->header_pixel);
                Configuration::updateValue(self::_MPrefix . 'WISER_CONTENT_NUDGE_' . $id_shop, htmlspecialchars($data->nudges_section));
                Configuration::updateValue(self::_MPrefix . 'WISER_CONTENT_RATING_' . $id_shop, htmlspecialchars($data->product_rating_count));
                Configuration::updateValue(self::_MPrefix . 'WISER_CONTENT_REVIEW_' . $id_shop, htmlspecialchars($data->product_review_section));

                $this->notification[$id_shop] = [
                    'status' => 'success',
                    'msg' => $this->trans($tokens->msg),
                ];
            } else {
                // Clear invalid values
                Configuration::updateValue(self::_MPrefix . 'WISER_REVIEW_TI_' . $id_shop, '');
                Configuration::updateValue(self::_MPrefix . 'WISER_AUTOMATION_ID_' . $id_shop, '');
                Configuration::updateValue(self::_MPrefix . 'WISER_HEADER_PXL_' . $id_shop, '');
                Configuration::updateValue(self::_MPrefix . 'WISER_CONTENT_NUDGE_' . $id_shop, '');
                Configuration::updateValue(self::_MPrefix . 'WISER_CONTENT_RATING_' . $id_shop, '');
                Configuration::updateValue(self::_MPrefix . 'WISER_CONTENT_REVIEW_' . $id_shop, '');

                $this->notification[$id_shop] = [
                    'status' => 'error',
                    'msg' => $this->trans($tokens->msg),
                ];
            }
        }
    }

    public function hookDisplaybackOfficeHeader() 
    {
        $this->context->controller->addJS(($this->_path).'views/js/back.js');
    }

    public function hookDisplayHeader() 
    {
        $id_shop = $this->context->shop->id;

        $live_mode = (boolean) Configuration::get(self::_MPrefix.'LIVE_MODE_'.$id_shop, true);
        
        $config = array(
            'liveMode' => $live_mode,
        );

        if(!$live_mode) 
            return;

        // define js variables
        Media::addJsDef([self::JS_VAR => $config]);

        $header_pxl = Configuration::get(self::_MPrefix.'WISER_HEADER_PXL_'.$id_shop, true);
        $this->context->smarty->assign('header_pxl', $header_pxl);
        
        $this->context->controller->addJS(($this->_path).'views/js/front.js');
        $this->context->controller->addCSS(($this->_path).'/views/css/front.css');

        return $this->fetch('module:wiserreview/views/templates/hook/script.tpl');
    }

    public function hookDisplayWiserReviewNudge()
    {
        $id_shop = $this->context->shop->id;
        $id_product = Tools::getValue('id_product');
        $content = Configuration::get(self::_MPrefix.'WISER_CONTENT_NUDGE_'.$id_shop, true);
        $product = new Product($id_product, false, $this->context->language->id);
        $sku = $product->reference;

        $count_review_div = str_replace(['{$product.id}', '{$product.skuid}'], [$id_product, $sku."'"], $content);
        // $count_review_div = str_replace('{$product.id}', $id_product, $content);
        echo htmlspecialchars_decode($count_review_div);
    }

    // hookDisplayProductListReviews
    public function hookDisplayWiserReviewProductListReviews($params)
    {
        $id_shop = $this->context->shop->id;
        $id_product = $params['product']['id_product'];
        if(empty($id_product)) {
            $id_product = $params['smarty']->tpl_vars['product']->value->id_product;
        }
        $content = Configuration::get(self::_MPrefix.'WISER_CONTENT_RATING_'.$id_shop, true);
        $product = new Product($id_product, false, $this->context->language->id);
        $sku = $product->reference;

        $count_review_div = str_replace(['{$product.id_product}', '{$product.skuid}'], [$id_product, $sku], $content);
        // $count_review_div = str_replace('{$product.id_product}', $id_product, $content);
        echo htmlspecialchars_decode($count_review_div);
        
    }

    public function hookDisplayWiserReviewFooterProduct()
    {
        $id_shop = $this->context->shop->id;
        $id_product = Tools::getValue('id_product');
        $content = Configuration::get(self::_MPrefix.'WISER_CONTENT_REVIEW_'.$id_shop, true);

        $product = new Product($id_product, false, $this->context->language->id);
        $sku = $product->reference;

        $count_review_div = str_replace(['{$product.id}', '{$product.skuid}'], [$id_product, $sku."'"], $content);
        echo htmlspecialchars_decode($count_review_div);
    }

    public function hookDisplayWiserReviewProductPriceBlock($params)
    {
        $id_shop = $this->context->shop->id;
        $id_product = Tools::getValue('id_product');
        if(empty($id_product)) {
            $id_product = $params['product']['id_product'];
        }

        if($this->context->controller->php_self == 'product') {
            $content = Configuration::get(self::_MPrefix.'WISER_CONTENT_RATING_'.$id_shop, true);
            $product = new Product($id_product, false, $this->context->language->id);
            $sku = $product->reference;

            $count_review_div = str_replace(['{$product.id_product}', '{$product.skuid}'], [$id_product, $sku], $content);
            // $count_review_div = str_replace('{$product.id_product}', $id_product, $content);
            echo htmlspecialchars_decode($count_review_div);
        }
    }

    public function hookDisplayProductAdditionalInfo() 
    {
        // Nudge
        $id_shop = $this->context->shop->id;
        $status = (bool) Configuration::get(self::_MPrefix.'REVIEW_NUDGES_'.$id_shop, true);
        if($status) {
            $this->hookDisplayWiserReviewNudge();
        }
    }
    
    public function hookDisplayProductListReviews($params) 
    {
        // Product Card
        $id_shop = $this->context->shop->id;
        $status = (bool) Configuration::get(self::_MPrefix.'COLLECTION_SECTION_'.$id_shop, true);
        if($status) {
            $this->hookDisplayWiserReviewProductListReviews($params);
        }
    }

    public function hookDisplayFooterProduct() 
    {
        // Product Page Bottom
        $id_shop = $this->context->shop->id;
        $status = (bool) Configuration::get(self::_MPrefix.'PRODUCT_REVIEW_SECTION_'.$id_shop, true);
        if($status) {
            $this->hookDisplayWiserReviewFooterProduct();
        }
    }

    public function hookDisplayProductPriceBlock($params) 
    {
        // Product Page Top
        $id_shop = $this->context->shop->id;
        $status = (bool) Configuration::get(self::_MPrefix.'PRODUCT_SECTION_'.$id_shop, true);
        if($status) {
            $this->hookDisplayWiserReviewProductPriceBlock($params);
        }
    }

    public function getProductRewrite($id_product)
    {
        $context = Context::getContext();
        $link = $context->link;

        $product = new Product((int)$id_product, false, $context->language->id, $context->shop->id);

        return $link->getProductLink($product);
    }

    public function hookActionValidateOrder($params) 
    {
        $order_data = $this->getFormattedOrderData($params['order']);

        $this->postOrderToWiser(json_encode($order_data));
    }

    public function getFormattedOrderData($order) 
    {
        $customer = new Customer($order->id_customer);
        $currency = new Currency($order->id_currency);
        $address = new Address($order->id_address_delivery); 
        $order_date = $order->date_add;
        
        $mobile_number = !empty($address->phone_mobile) ? $address->phone_mobile : $address->phone;
        $order_data = [
            'oid' => (int) $order->id,
            'cdt' => $order_date,
            'customer' => [
                'email' => $customer->email,
                'phone' => $mobile_number,
                'first_name' => $customer->firstname,
                'last_name' => $customer->lastname,
            ],
            'line_items' => [],
        ];

        $products = $order->getProducts();

        foreach ($products as $product) {
            $product_url = Context::getContext()->link->getProductLink($product['id_product']);
            $image_id = Product::getCover($product['id_product']);
            $link_rewrite = $this->getProductRewrite($product['id_product']);

            $image_url = $image_id ? Context::getContext()->link->getImageLink($link_rewrite, $image_id['id_image'], 'home_default') : '';

            $order_data['line_items'][] = [
                'id' => (int) $product['id_product'],
                'pn' => $product['product_name'],
                'sku'=> $product['reference'],
                'prc' => (float) $product['unit_price_tax_incl'],
                'pu' => $product_url,
                'piu' => $image_url,
            ];
        }

        return $order_data;
    }

    public function postOrderToWiser($order)
    {   
        $id_shop = $this->context->shop->id;
        $ti = Configuration::get(self::_MPrefix.'WISER_REVIEW_TI_'.$id_shop, true);
        $atmid = Configuration::get(self::_MPrefix.'WISER_AUTOMATION_ID_'.$id_shop, true);

        $url = "https://api.wiserreview.com/api/prestashop/webhook?wsid=".$ti."&atmid=".$atmid;
        
        $ch = curl_init($url);
        
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $order);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json'
        ]);

        $response = curl_exec($ch);
        
        if (curl_errno($ch)) {
            PrestaShopLogger::addLog('cURL Error: ' . curl_error($ch), 3);
        }
        
        curl_close($ch);

        PrestaShopLogger::addLog('Order Sync Response: ' . $response, 1);

        return true;
    }
    
    private function verifyApiTokey($key)
    {
        if(empty($key))
            return;

        $url = "https://api.wiserreview.com/api/prestashop/verify?key=".$key;

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPGET, true);

        $response = curl_exec($ch);
        
        if (curl_errno($ch)) {
            return false;
        }

        curl_close($ch);

        $response = json_decode($response);
        
        return $response;
    }

}
